<?php

require '../vendor/autoload.php';

use Dotenv\Dotenv;

// Cargar las variables de entorno desde la raíz del proyecto
$dotenv = Dotenv::createImmutable(__DIR__ . '/../'); // Subir un nivel para acceder a la raíz
$dotenv->load();

define('SYSTEMNAME', 'BENDEY_AGUA');
define('ZONA_HORARIA', $_ENV['APP_TIMEZONE'] ?? 'America/Lima');

// Configurar zona horaria
date_default_timezone_set(ZONA_HORARIA);

// BASE_URL dinámico que funciona tanto en local como en producción
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'];

// Para desarrollo local, usar la URL del .env si está disponible
if (isset($_ENV['APP_URL']) && !empty($_ENV['APP_URL'])) {
    $baseUrl = rtrim($_ENV['APP_URL'], '/') . '/';
} else {
    // Fallback para cuando no hay APP_URL definida
    $baseUrl = $protocol . '://' . $host . '/';
}

$_SERVER['BASE_URL'] = $baseUrl;
define('BASE_URL', $baseUrl);

// Configuración de ambiente usando variables de entorno
define('APP_ENV', $_ENV['APP_ENV'] ?? 'production');
define('APP_NAME', $_ENV['APP_NAME'] ?? 'Bendey Agua');
define('DEBUG_MODE', filter_var($_ENV['APP_DEBUG'] ?? false, FILTER_VALIDATE_BOOLEAN));

// ============================================
// DEBUG COMPLETO - SOLO EN MODO DESARROLLO
// ============================================

if (APP_ENV === 'local' && DEBUG_MODE) {
    register_shutdown_function('showDebugPanel');
}

function showDebugPanel($protocol=null) {
    // Aquí mueve TODO tu código HTML/CSS/JS del debug
    // 1. INFORMACIÓN DEL SISTEMA Y SERVIDOR
    $debugInfo = [
        'APLICACIÓN' => [
            'Nombre' => APP_NAME,
            'Sistema' => SYSTEMNAME,
            'Ambiente' => APP_ENV,
            'Debug Activo' => DEBUG_MODE ? 'Sí' : 'No',
            'Zona Horaria' => ZONA_HORARIA,
            'Fecha/Hora Actual' => date('Y-m-d H:i:s'),
            'Timestamp' => time()
        ],
        'SERVIDOR' => [
            'PHP Version' => phpversion(),
            'Servidor' => $_SERVER['SERVER_SOFTWARE'] ?? 'No disponible',
            'Host' => $_SERVER['HTTP_HOST'] ?? 'No disponible',
            'Puerto' => $_SERVER['SERVER_PORT'] ?? 'No disponible',
            'Protocolo' => $protocol,
            'IP del Servidor' => $_SERVER['SERVER_ADDR'] ?? 'No disponible',
            'Sistema Operativo' => php_uname()
        ],
        'URLS Y RUTAS' => [
            'BASE_URL' => BASE_URL,
            'APP_URL (.env)' => $_ENV['APP_URL'] ?? 'No definida',
            'Document Root' => $_SERVER['DOCUMENT_ROOT'] ?? 'No disponible',
            'Script Name' => $_SERVER['SCRIPT_NAME'] ?? 'No disponible',
            'Request URI' => $_SERVER['REQUEST_URI'] ?? 'No disponible',
            'Directorio Actual' => __DIR__,
            'Archivo Actual' => __FILE__
        ],
        'VARIABLES DE ENTORNO (.env)' => [
            'APP_ENV' => $_ENV['APP_ENV'] ?? 'No definida',
            'APP_NAME' => $_ENV['APP_NAME'] ?? 'No definida',
            'APP_DEBUG' => $_ENV['APP_DEBUG'] ?? 'No definida',
            'APP_URL' => $_ENV['APP_URL'] ?? 'No definida',
            'APP_TIMEZONE' => $_ENV['APP_TIMEZONE'] ?? 'No definida',
            'DB_CONNECTION' => $_ENV['DB_CONNECTION'] ?? 'No definida',
            'DB_HOST' => $_ENV['DB_HOST'] ?? 'No definida',
            'DB_PORT' => $_ENV['DB_PORT'] ?? 'No definida',
            'DB_DATABASE' => $_ENV['DB_DATABASE'] ?? 'No definida',
            'DB_USERNAME' => $_ENV['DB_USERNAME'] ?? 'No definida',
            'DB_PASSWORD' => isset($_ENV['DB_PASSWORD']) ? '***OCULTA***' : 'No definida'
        ],
        'MEMORIA Y RECURSOS' => [
            'Límite de Memoria' => ini_get('memory_limit'),
            'Memoria Usada' => round(memory_get_usage() / 1024 / 1024, 2) . ' MB',
            'Memoria Pico' => round(memory_get_peak_usage() / 1024 / 1024, 2) . ' MB',
            'Tiempo Límite' => ini_get('max_execution_time') . ' segundos',
            'Tamaño Max Upload' => ini_get('upload_max_filesize'),
            'Max Post Size' => ini_get('post_max_size')
        ],
        'EXTENSIONES PHP' => [
            'PDO' => extension_loaded('pdo') ? 'Sí' : 'No',
            'MySQLi' => extension_loaded('mysqli') ? 'Sí' : 'No',
            'cURL' => extension_loaded('curl') ? 'Sí' : 'No',
            'GD' => extension_loaded('gd') ? 'Sí' : 'No',
            'JSON' => extension_loaded('json') ? 'Sí' : 'No',
            'OpenSSL' => extension_loaded('openssl') ? 'Sí' : 'No',
            'Zip' => extension_loaded('zip') ? 'Sí' : 'No'
        ],
        'REQUEST INFO' => [
            'Método' => $_SERVER['REQUEST_METHOD'] ?? 'No disponible',
            'User Agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'No disponible',
            'IP Cliente' => $_SERVER['REMOTE_ADDR'] ?? 'No disponible',
            'Referrer' => $_SERVER['HTTP_REFERER'] ?? 'No disponible',
            'Accept' => $_SERVER['HTTP_ACCEPT'] ?? 'No disponible',
            'Accept Language' => $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? 'No disponible'
        ]
    ];

    // 2. ESCRIBIR EN LOG DE ERRORES
    error_log("=== BENDEY AGUA DEBUG COMPLETO ===");
    error_log("Timestamp: " . date('Y-m-d H:i:s'));
    foreach ($debugInfo as $seccion => $datos) {
        error_log("--- $seccion ---");
        foreach ($datos as $clave => $valor) {
            error_log("$clave: $valor");
        }
    }
    error_log("=== FIN DEBUG ===");

    // 3. COMENTARIOS HTML PARA DESARROLLADORES
    echo "<!-- ================================= -->\n";
    echo "<!-- BENDEY AGUA - DEBUG INFORMACIÓN -->\n";
    echo "<!-- ================================= -->\n";
    foreach ($debugInfo as $seccion => $datos) {
        echo "<!-- $seccion -->\n";
        foreach ($datos as $clave => $valor) {
            echo "<!-- $clave: $valor -->\n";
        }
    }
    echo "<!-- ================================= -->\n";

    // 4. PANEL VISUAL DE DEBUG PARA EL NAVEGADOR
    echo "<div id='bendey-debug-panel' style='
        position: fixed;
        top: 10px;
        right: 10px;
        width: 350px;
        max-height: 80vh;
        overflow-y: auto;
        background: #1a1a1a;
        color: #00ff00;
        font-family: monospace;
        font-size: 11px;
        border: 2px solid #00ff00;
        border-radius: 5px;
        z-index: 9999;
        padding: 10px;
        box-shadow: 0 0 20px rgba(0,255,0,0.3);
    '>";
    
    echo "<div style='text-align: center; margin-bottom: 10px; color: #ffff00;'>";
    echo "<strong>🔧 BENDEY AGUA DEBUG</strong>";
    echo "<button onclick='document.getElementById(\"bendey-debug-panel\").style.display=\"none\"' style='float: right; background: red; color: white; border: none; padding: 2px 8px; cursor: pointer;'>✕</button>";
    echo "</div>";

    foreach ($debugInfo as $seccion => $datos) {
        echo "<div style='margin-bottom: 15px;'>";
        echo "<div style='background: #333; padding: 5px; margin-bottom: 5px; font-weight: bold; color: #ffff00;'>$seccion</div>";
        
        foreach ($datos as $clave => $valor) {
            $valorMostrar = is_bool($valor) ? ($valor ? 'true' : 'false') : htmlspecialchars($valor);
            echo "<div style='padding: 2px 0; border-bottom: 1px solid #333;'>";
            echo "<span style='color: #00ffff;'>$clave:</span> ";
            echo "<span style='color: #ffffff;'>$valorMostrar</span>";
            echo "</div>";
        }
        echo "</div>";
    }
    
    // Botón para mostrar $_SERVER completo
    echo "<button onclick='toggleServerInfo()' style='
        background: #0066cc; 
        color: white; 
        border: none; 
        padding: 5px 10px; 
        cursor: pointer; 
        margin: 5px 0;
        width: 100%;
    '>Ver $_SERVER Completo</button>";
    
    echo "<div id='server-info' style='display: none; background: #2a2a2a; padding: 10px; max-height: 200px; overflow-y: auto;'>";
    foreach ($_SERVER as $key => $value) {
        // Manejar diferentes tipos de datos
        if (is_array($value)) {
            $valorMostrar = '[Array] ' . json_encode($value, JSON_UNESCAPED_UNICODE);
        } elseif (is_object($value)) {
            $valorMostrar = '[Object] ' . get_class($value);
        } elseif (is_bool($value)) {
            $valorMostrar = $value ? 'true' : 'false';
        } elseif (is_null($value)) {
            $valorMostrar = 'null';
        } else {
            $valorMostrar = (string)$value;
        }
        
        echo "<div><span style='color: #00ffff;'>$key:</span> <span style='color: #ffffff;'>" . htmlspecialchars($valorMostrar) . "</span></div>";
    }
    echo "</div>";

    echo "</div>";

    // 5. JAVASCRIPT PARA FUNCIONALIDAD DEL PANEL
    echo "<script>
    function toggleServerInfo() {
        var serverInfo = document.getElementById('server-info');
        serverInfo.style.display = serverInfo.style.display === 'none' ? 'block' : 'none';
    }
    
    // Hacer el panel arrastrable
    var panel = document.getElementById('bendey-debug-panel');
    var isDragging = false;
    var currentX, currentY, initialX, initialY;
    
    panel.addEventListener('mousedown', function(e) {
        if (e.target.tagName !== 'BUTTON') {
            isDragging = true;
            initialX = e.clientX - panel.offsetLeft;
            initialY = e.clientY - panel.offsetTop;
        }
    });
    
    document.addEventListener('mousemove', function(e) {
        if (isDragging) {
            e.preventDefault();
            currentX = e.clientX - initialX;
            currentY = e.clientY - initialY;
            panel.style.left = currentX + 'px';
            panel.style.top = currentY + 'px';
            panel.style.right = 'auto';
        }
    });
    
    document.addEventListener('mouseup', function() {
        isDragging = false;
    });
    
    // Mostrar notificación en consola
    console.log('%c🔧 BENDEY AGUA DEBUG ACTIVO', 'background: #1a1a1a; color: #00ff00; padding: 10px; font-size: 16px;');
    console.log('Panel de debug visible en la esquina superior derecha');
    </script>";

    // 6. MENSAJE EN CONSOLA DEL NAVEGADOR
    echo "<script>
    console.group('🔧 BENDEY AGUA - DEBUG INFO');
    ";
    
    foreach ($debugInfo as $seccion => $datos) {
        echo "console.group('$seccion');";
        foreach ($datos as $clave => $valor) {
            $valorJS = is_bool($valor) ? ($valor ? 'true' : 'false') : json_encode($valor);
            echo "console.log('$clave:', $valorJS);";
        }
        echo "console.groupEnd();";
    }
    
    echo "
    console.groupEnd();
    </script>";

    // 7. VERIFICAR ARCHIVOS IMPORTANTES
    $archivosImportantes = [
        '../.env' => 'Archivo de configuración',
        '../composer.json' => 'Dependencias Composer',
        '../vendor/autoload.php' => 'Autoloader Composer'
    ];

    echo "<div style='background: #fffacd; padding: 10px; margin: 10px 0; border-left: 4px solid #ff6b6b;'>";
    echo "<h4 style='margin: 0 0 10px 0; color: #333;'>📁 Verificación de Archivos</h4>";
    foreach ($archivosImportantes as $archivo => $descripcion) {
        $existe = file_exists($archivo);
        $estado = $existe ? '✅ Existe' : '❌ No encontrado';
        $color = $existe ? 'green' : 'red';
        echo "<div style='color: $color;'>$descripcion ($archivo): $estado</div>";
    }
    echo "</div>";
}
?>